/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.model.MimeType;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MANIFEST_FILENAME = "Manifest";
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    public static final String SIGNATURE_FILENAME = "signature";
    public static final String TIMESTAMP_FILENAME = "timestamp";
    public static final String TST_EXTENSION = ".tst";
    public static final String XML_EXTENSION = ".xml";
    public static final String ZIP_ENTRY_DETACHED_FILE = "detached-file";
    private static final long ZIP_ENTRY_THRESHOLD = 1000000L;
    private static final long ZIP_ENTRY_RATIO = 50L;
    private static final int MAX_MALFORMED_FILES = 100;

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(SIGNATURE_FILENAME) && !entryName.contains(MANIFEST_FILENAME);
    }

    public static boolean isTimestamp(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(TIMESTAMP_FILENAME) && entryName.endsWith(TST_EXTENSION);
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        String asicParameterMimeType = asicParameters.getMimeType();
        String mimeTypeString = Utils.isStringBlank(asicParameterMimeType) ? (ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE.getMimeTypeString() : MimeType.ASICS.getMimeTypeString()) : asicParameterMimeType;
        return mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType asicMimeType) {
        return MimeType.ASICS.equals(asicMimeType) || MimeType.ASICE.equals(asicMimeType);
    }

    public static boolean isOpenDocumentMimeType(MimeType mimeType) {
        return MimeType.ODT.equals(mimeType) || MimeType.ODS.equals(mimeType) || MimeType.ODG.equals(mimeType) || MimeType.ODP.equals(mimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        if (MimeType.ASICS.equals(asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeType.ASICE.equals(asicMimeType) || ASiCUtils.isOpenDocumentMimeType(asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype '" + asicMimeType.getMimeTypeString() + "'");
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        return ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE : MimeType.ASICS;
    }

    public static boolean isArchiveContainsCorrectSignatureFileWithExtension(DSSDocument container, String extension) {
        List<String> filenames = ASiCUtils.getFileNames(container);
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean isArchiveContainsCorrectTimestamp(DSSDocument container) {
        List<String> filenames = ASiCUtils.getFileNames(container);
        for (String filename : filenames) {
            if (!ASiCUtils.isTimestamp(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isArchive(List<DSSDocument> docs) {
        if (Utils.collectionSize(docs) == 1) {
            return ASiCUtils.isZip(docs.get(0));
        }
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isZip(DSSDocument dssDocument) {
        if (dssDocument == null) {
            return false;
        }
        byte[] preamble = new byte[2];
        try (InputStream is = dssDocument.openStream();){
            int r = is.read(preamble, 0, 2);
            if (r != 2) {
                boolean bl = false;
                return bl;
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to read the 2 first bytes", e);
        }
        if (preamble[0] != 80) return false;
        if (preamble[1] != 75) return false;
        return true;
    }

    public static boolean isASiCWithCAdES(DSSDocument archive) {
        return ASiCUtils.isArchiveContainsCorrectSignatureFileWithExtension(archive, ".p7s") || ASiCUtils.isArchiveContainsCorrectTimestamp(archive);
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(".p7s");
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDoc) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDoc);
        if (mimeTypeDoc != null) {
            return ASiCUtils.isOpenDocumentMimeType(mimeType);
        }
        return false;
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray(is);
                String mimeTypeString = new String(byteArray, StandardCharsets.UTF_8);
                mimeType = MimeType.fromMimeTypeString(mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException(e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static ASiCContainerType getContainerType(DSSDocument archive, DSSDocument mimetype, String zipComment, List<DSSDocument> signedDocuments) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeType(archive.getMimeType());
        if (containerType == null && (containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetype)) == null) {
            containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        }
        if (containerType == null) {
            LOG.warn("Unable to define the ASiC Container type with its properties");
            if (Utils.collectionSize(signedDocuments) == 1) {
                containerType = ASiCContainerType.ASiC_S;
            } else if (Utils.collectionSize(signedDocuments) > 1) {
                containerType = ASiCContainerType.ASiC_E;
            } else {
                throw new DSSException("The provided file does not contain signed documents. The signature validation is not possible");
            }
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank(zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString(asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static String getPadNumber(int num) {
        String numStr = String.valueOf(num);
        String zeroPad = "000";
        return zeroPad.substring(numStr.length()) + numStr;
    }

    public static boolean isAsic(List<DSSDocument> documents) {
        if (ASiCUtils.isArchive(documents)) {
            DSSDocument archive = documents.get(0);
            boolean cades = ASiCUtils.isArchiveContainsCorrectSignatureFileWithExtension(archive, "p7s");
            boolean xades = ASiCUtils.isArchiveContainsCorrectSignatureFileWithExtension(archive, "xml");
            boolean timestamp = ASiCUtils.isArchiveContainsCorrectTimestamp(archive);
            return cades || xades || timestamp;
        }
        return false;
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.contains("ASiCArchiveManifest") && fileName.endsWith(XML_EXTENSION);
    }

    public static void secureCopy(InputStream is, OutputStream os, long containerSize) throws IOException {
        int nRead;
        byte[] data = new byte[2048];
        int byteCounter = 0;
        long allowedSize = containerSize * 50L;
        while ((nRead = is.read(data)) != -1) {
            if ((long)(byteCounter += nRead) > 1000000L && (long)byteCounter > allowedSize) {
                throw new DSSException("Zip Bomb detected in the ZIP container. Validation is interrupted.");
            }
            os.write(data, 0, nRead);
        }
    }

    public static List<String> getFileNames(DSSDocument archive) {
        ArrayList<String> filenames = new ArrayList<String>();
        try (InputStream is = archive.openStream();
             ZipInputStream zis = new ZipInputStream(is);){
            ZipEntry entry;
            while ((entry = ASiCUtils.getNextValidEntry(zis)) != null) {
                filenames.add(entry.getName());
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to retrieve filenames", e);
        }
        return filenames;
    }

    public static ZipEntry getNextValidEntry(ZipInputStream zis) {
        for (int counter = 0; counter < 100; ++counter) {
            try {
                return zis.getNextEntry();
            }
            catch (Exception e) {
                LOG.warn("ZIP container contains a malformed, corrupted or not accessible entry! The entry is skipped. Reason: [{}]", (Object)e.getMessage());
                ASiCUtils.closeEntry(zis);
                continue;
            }
        }
        throw new DSSException("Unable to retrieve a valid ZipEntry (100 tries)");
    }

    public static DSSDocument getCurrentDocument(String filepath, ZipInputStream zis, long containerSize) throws IOException {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            ASiCUtils.secureCopy(zis, baos, containerSize);
            baos.flush();
            InMemoryDocument inMemoryDocument = new InMemoryDocument(baos.toByteArray(), filepath);
            return inMemoryDocument;
        }
    }

    public static void closeEntry(ZipInputStream zis) {
        try {
            zis.closeEntry();
        }
        catch (IOException e) {
            throw new DSSException("Unable to close entry", e);
        }
    }

    public static String getNextASiCEManifestName(String expectedManifestName, List<DSSDocument> existingManifests) {
        List<String> manifestNames = ASiCUtils.getDSSDocumentNames(existingManifests);
        String manifestName = null;
        for (int i = 0; i < existingManifests.size() + 1; ++i) {
            String suffix = i == 0 ? "" : String.valueOf(i);
            manifestName = META_INF_FOLDER + expectedManifestName + suffix + XML_EXTENSION;
            if (ASiCUtils.isValidName(manifestName, manifestNames)) break;
        }
        return manifestName;
    }

    public static List<String> getDSSDocumentNames(List<DSSDocument> documents) {
        return documents.stream().map(DSSDocument::getName).collect(Collectors.toList());
    }

    private static boolean isValidName(String name, List<String> notValidNames) {
        return !notValidNames.contains(name);
    }

    public static boolean isASiCSArchive(DSSDocument document) {
        return Utils.areStringsEqual(PACKAGE_ZIP, document.getName());
    }

    public static List<DSSDocument> getPackageZipContent(DSSDocument packageZip) {
        ArrayList<DSSDocument> result = new ArrayList<DSSDocument>();
        long containerSize = DSSUtils.getFileByteSize(packageZip);
        try (InputStream is = packageZip.openStream();
             ZipInputStream packageZipInputStream = new ZipInputStream(is);){
            ZipEntry entry;
            while ((entry = ASiCUtils.getNextValidEntry(packageZipInputStream)) != null) {
                result.add(ASiCUtils.getCurrentDocument(entry.getName(), packageZipInputStream, containerSize));
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to extract package.zip", e);
        }
        return result;
    }
}

